<?php


namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    // List users
    public function index()
    {
        $users = User::with('roles')->whereDoesntHave('roles', function ($query) {
        $query->where('id', 1);
        })->get();
        return view('user.index', compact('users'));
    }

    // Show create user form
    public function create()
    {
        $roles = Role::pluck('name', 'id');
        return view('user.create', compact('roles'));
    }

    // Store new user
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8',
            'role' => 'required|exists:roles,id',
            'image' => 'required',
            
        ]);
        
    // Handle profile photo upload
    if ($request->hasFile('image')) {
        $imageName = time() . '.' . $request->file('image')->extension();
        $request->file('image')->move(public_path('profile_photos'), $imageName);
    } else {
        $imageName = null;
    }
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
             'image' => $imageName, 
        ]);

        $role = Role::findById($request->role);
        $user->assignRole($role);

        return redirect()->route('user.list')->with('success', 'User created successfully.');
    }

    // Show edit user form
    public function edit(User $user)
    {
        $roles = Role::pluck('name', 'id');
        return view('user.edit', compact('user', 'roles'));
    }

    // Update user
  public function update(Request $request, User $user)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $user->id,
        'role' => 'required|exists:roles,id',
        'image' => 'nullable|image', // Image is optional
    ]);

    // Handle image upload only if a new image is provided
    if ($request->hasFile('image')) {
        $imageName = time() . '.' . $request->file('image')->extension();
        $request->file('image')->move(public_path('profile_photos'), $imageName);
    } else {
        $imageName = $user->image; // Keep the old image if no new image is uploaded
    }

    // Only update password if provided
    $updateData = [
        'name' => $request->name,
        'email' => $request->email,
        'image' => $imageName, 
    ];

    if ($request->filled('password')) {
        $updateData['password'] = Hash::make($request->password);
    }

    $user->update($updateData);

    $role = Role::findById($request->role);
    $user->syncRoles([$role]);

    return redirect()->route('user.list')->with('success', 'User updated successfully.');
}


    // Delete user
    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('user.list')->with('success', 'User deleted successfully.');
    }
}
