<?php

namespace App\Http\Controllers;

use App\Models\CarHandover;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

use Barryvdh\DomPDF\Facade\Pdf;


class CarHandoverController extends Controller
{
   

   // Show list of handovers
    public function index()
    {
        $handovers = CarHandover::latest()->get();
        return view('CarHandover.index', compact('handovers'));
    }
    
    public function create()
    {
        return view('CarHandover.create');
    }

    public function store(Request $request)
    {
         $validator = Validator::make($request->all(),[
            'plate_number' => 'required|string|max:255',
            'model' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'brand' => 'required|string|max:255',
            'mileage' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'receiver_name' => 'required|string|max:255',
            'receiver_phone' => 'required|string|max:20',
            'receiver_id' => 'required|string|max:50',
            'supervisor_name' => 'required|string|max:255',
            'handover_date' => 'required|date',
            'handover_time' => 'required',
            'car_image' => 'required|image',
            
            'delegated' => 'required|in:0,1',
            'delegation_number' => 'nullable|string',
            'reason_no_delegation' => 'nullable|string',
            
            
            'contents' => 'required|min:1',
            'contents.*' => 'string',
            'other_contents' => 'required_if:contents.*,"Other"|string|nullable',
        ]);
      
    // Add conditional validation
    if ($request->input('delegated') == 1) {
        $validator->addRules([
            'delegation_number' => 'required|string',
        ]);
    } elseif ($request->input('delegated') == 0) {
        $validator->addRules([
            'reason_no_delegation' => 'required|string',
        ]);
    }
        
        $validator->validate();
        
        $data = $request->contents; // This is already an array

 // Process contents
    $contents = $request->contents;
    if (in_array('Other', $contents)) {
        if (!$request->filled('other_contents')) {
            return back()->withErrors(['other_contents' => 'The other contents field is required.'])->withInput();
        }
        $contents = array_filter($contents, fn($item) => $item !== 'Other');
        $contents[] = ['other' => $request->other_contents];
    }


        $handover = new CarHandover();
        $handover->plate_number = $request->plate_number;
        $handover->model = $request->model;
        $handover->type = $request->type;
        $handover->brand = $request->brand;
        $handover->mileage = $request->mileage;
        $handover->location = $request->location;
        
        $handover->delegated = $request->delegated;
        $handover->reason_for_no_delegation = $request->reason_no_delegation;
        $handover->delegation_number = $request->delegation_number;
        
    
        $handover->vehicleContents = $contents;
      

        $handover->receiver_name = $request->receiver_name;
        $handover->receiver_phone = $request->receiver_phone;
        $handover->receiver_id = $request->receiver_id;
        $handover->supervisor_name = $request->supervisor_name;
        $handover->handover_date = $request->handover_date;
        $handover->handover_time = $request->handover_time;

       
        
        if ($request->hasFile('car_image')) {
            $imageName = time() . '.' . $request->car_image->extension();
            $request->car_image->move(public_path('car_images'), $imageName);
            $handover->car_image = $imageName;
        } else {
            $handover->car_image = null;
        }


        $handover->save();

        return redirect()->back()->with('success', 'Car Handover Recorded Successfully!');
    }


    
        public function edit($id)
    {
        $handover = CarHandover::findOrFail($id);
        return view('CarHandover.edit', compact('handover'));
    }
    
  public function update(Request $request, $id)
{
    $request->validate([
        'plate_number' => 'required|string',
        'model' => 'nullable|string',
        'type' => 'nullable|string',
        'brand' => 'nullable|string',
        'mileage' => 'nullable|string',
        'location' => 'required|string',
        'receiver_name' => 'nullable|string',
        'receiver_phone' => 'nullable|string',
        'receiver_id' => 'nullable|string',
        'supervisor_name' => 'nullable|string',
        'handover_date' => 'nullable|date',
        'handover_time' => 'nullable',
        'contents' => 'array|nullable',
        'car_image' => 'nullable|image',
    ]);

    $handover = CarHandover::findOrFail($id);
    $handover->fill($request->except(['car_image', 'contents', 'other_contents']));

    $handover->delegated = $request->delegated;
    $handover->delegation_number = $request->delegated ? $request->delegation_number : null;
    $handover->reason_for_no_delegation = !$request->delegated ? $request->reason_no_delegation : null;

    // ✅ Process vehicle contents like in store()
    $contents = $request->contents ?? [];

    if (in_array('Other', $contents)) {
        if (!$request->filled('other_contents')) {
            return back()->withErrors(['other_contents' => 'The other contents field is required.'])->withInput();
        }
        $contents = array_filter($contents, fn($item) => $item !== 'Other');
        $contents[] = ['other' => $request->other_contents];
    }

    $handover->vehicleContents = $contents;

    // ✅ Handle image
    if ($request->hasFile('car_image')) {
        $path = $request->file('car_image')->store('car_images', 'public');
        $handover->car_image = $path;
    }

    $handover->save();

    return redirect()->route('handover.List')->with('success', 'Car handover updated successfully.');
}



    
    public function show($id)
    {
        $handover = CarHandover::findOrFail($id);
        return view('CarHandover.show', compact('handover'));
    }
 
    public function destroy($id)
    {
        CarHandover::destroy($id);
        return back()->with('success', 'Car handover deleted.');
    }
    
    
    
      public function downloadPdf()
    {
        $handovers = CarHandover::all();
        $pdf = Pdf::loadView('CarHandover.pdf', compact('handovers'));
        return $pdf->download('car_handover_report_list.pdf');
    }
}

