<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail; 
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Exception;
use Carbon\Carbon;
use App\Mail\SendOtp;
use Illuminate\Support\Facades\Cache;



class AuthController extends Controller
{
    public function register(Request $request)
    {
       
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'mobile' => 'required',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string',
        ]);
    
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422); 
        }
    
        $user = User::create([
            'name' => $request->name,
            'mobile' => $request->mobile,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);
        
        $token = $user->createToken('auth_token')->plainTextToken;
        
        return response()->json([
            'success' => true,
            'message' => 'User registered successfully.',
            'data' => [
                'user' => $user,
                'token' => $token,
                'token_type' => 'Bearer',
            ],
        ], 201);
    }

    public function login(Request $request)
    {
        // Validate incoming request data
        $validatedData = $request->validate([
            'email' => 'required|email|exists:users,email',
            'password' => 'required',
        ]);
    
        // Find the user by email
        $user = User::where('email', $validatedData['email'])->first();
    
        // Check if the password matches
        if ($user && Hash::check($validatedData['password'], $user->password)) {
            // Log in the user
            Auth::login($user);
    
            // Create a personal access token
            $token = $user->createToken('Dashboard')->plainTextToken;
    
            return response()->json([
                'success' => true,
                'token' => $token,
                'token_type' => 'Bearer Token',
            ], Response::HTTP_OK);
        }
    
        // If password does not match
        return response()->json([
            'success' => false,
            'message' => 'Invalid credentials.',
        ], Response::HTTP_UNAUTHORIZED);
    }

    public function Logout(Request $request)
    {
       
        $user = Auth::user();

       
        $user->tokens()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Successfully logged out',
        ], Response::HTTP_OK);
    }
       
    public function forgotPassword(Request $request)
    {
        try {
          
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|exists:users,email',
            ]);
    
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
            }
           
            $user = User::where('email', $request->email)->first();
            if (!$user) {
                return response()->json(['status' => false, 'error' => 'User not registered'], Response::HTTP_UNAUTHORIZED);
            }
            
            $otp = mt_rand(100000, 999999);
            $expiresAt = Carbon::now()->addMinutes(10);
    
            Cache::put('otp_' . $user->id, $otp, $expiresAt);
    
              Mail::to($request->email)->send(new SendOtp($otp));
       
            return response()->json(['status' => true, 'message' => 'OTP sent to your email successfully.'], Response::HTTP_OK);
    
        } catch (Exception $e) {
            Log::error('Forgot Password error: ' . $e->getMessage());
            return response()->json(['status' => false, 'error' => 'Something went wrong, please try again later.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
    
    public function verifyOtp(Request $request)
    {
        try {
            // Validate incoming request
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|exists:users,email',  // Ensure email exists in the system
                'otp' => 'required|digits:6',  // OTP should be a 6-digit number
                'password' => 'required|min:8|max:16',  // Password validation (confirmation and length)
            ]);
    
            // If validation fails, return errors
            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], Response::HTTP_UNPROCESSABLE_ENTITY);
            }
    
            // Retrieve user by email
            $user = User::where('email', $request->email)->first();
            if (!$user) {
                return response()->json(['status' => false, 'error' => 'User not found'], Response::HTTP_UNAUTHORIZED);
            }
    
            // Retrieve OTP from cache
            $cachedOtp = Cache::get('otp_' . $user->id);
    
            // If OTP does not exist or is incorrect, return an error
            if (!$cachedOtp || $cachedOtp != $request->otp) {
                return response()->json(['status' => false, 'error' => 'Invalid or expired OTP'], Response::HTTP_UNAUTHORIZED);
            }
    
            // Remove OTP from cache after successful verification
            Cache::forget('otp_' . $user->id);
    
            // Update the user's password
            $user->password = Hash::make($request->password);
            $user->save();  // Save the updated password
    
            return response()->json(['status' => true, 'message' => 'Password has been successfully updated'], Response::HTTP_OK);
    
        } catch (Exception $e) {
            Log::error('OTP verification and password update error: ' . $e->getMessage());
            return response()->json(['status' => false, 'error' => 'Something went wrong, please try again later.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

}
